# ============================================== #
# Mixture-based nonparametric density estimation #
#                                                #
# Author: Yong Wang (yongwang@auckland.ac.nz)    #
#         Department of Statistics               #
#         University of Auckland                 #
#         New Zealand                            #
# ============================================== #

# require(nspmix) 

mde = function(data, bwf=5:1/5, tol=1e-4,
               criterion=c("AICc","AIC","BIC"), 
               exhaustive=FALSE, verbose=0,
               plot=c("null","density","gradient"), ...) {
  criterion = match.arg(criterion)
  plot = match.arg(plot)
  bwf = unique(sort(bwf))
  k = length(bwf)
  ic = double(k)
  icmin = Inf
  names(icmin) = criterion
  n = length(data)
  s = sd(data)
  x = npnorm(data)
  info = matrix(nrow=k, ncol=3, dimnames=list(1:k,c("bwf", "ll", criterion)))
  for(i in 1:k) {
    r = cnm(x, init=list(beta=s*bwf[k-i+1]), tol=tol, ...)
    p = 2 * length(r$mix$pt) - 1    # number of parameters
    ic[i] = -2 * r$ll +
      switch(criterion,
             AICc = {2 * p + 2 * p * (p + 1) / max(n - p - 1, 0)},
             AIC = {2 * p},
             BIC = {log(n) * p}
             )
    info[i,] = c(bwf[k-i+1], r$ll, ic[i])
    if(ic[i] < icmin) {
      icmin[1] = ic[i]
      imin = i
      h = s * bwf[k-i+1]
      mix = r$mix
      ll = r$ll
      maxgrad = r$max.gradient
    }
    if(plot == "density") plot(x, r$mix, r$beta)
    if(plot == "gradient") plotgrad(x, r$mix, r$beta)
    if(verbose >= 1)
      cat("Iter.#", i, ":\tbwf = ", sprintf("%.2f", bwf[k-i+1]), "\tll = ",
          r$ll, "\t", criterion, " = ", ic[i], "\n", sep="")
    if(verbose >= 2) print(r$mix)
    if(!exhaustive && ic[i] > icmin + 10) break
  }
  fit = list(info=info[1:i,], s=s, maxgrad=maxgrad, ll=ll, criterion=icmin,
             bwf=bwf[k-imin+1], h=h, mix=mix)
  class(fit) = "mde"
  fit
}

plot.mde = function(x, data, components=c("null","proportions","curves"), ...) {
  components = match.arg(components)
  if(missing(data))
    plot(npnorm(numeric(0)), x$mix, x$h, components=components, ...)
  else plot(npnorm(data), x$mix, x$h, components=components, ...)
  invisible(x)
}

predict.mde = function(x, newdata, type=c("pdf","cdf"), lower.tail=TRUE,
                       log=FALSE) {
  type = match.arg(type)
  switch(type,
         pdf = dnpnorm(newdata, x$mix, x$h, log=log),
         cdf = pnpnorm(newdata, x$mix, x$h, lower.tail=lower.tail, log=log) )
}

