# =========================================================== #
# Mixture-based multivariate nonparametric density estimation #
#                                                             #
# Author: Yong Wang (yongwang@auckland.ac.nz)                 #
#         Department of Statistics                            #
#         University of Auckland                              #
#         New Zealand                                         #
# =========================================================== #

mmde = function(data, bwf=5:1/5, tol=1e-4,
                criterion=c("AICc","AIC","BIC"),
                exhaustive=FALSE, verbose=0, by,
                plot=c("null","marginals","density","components","gradient","pairs"),
                ...) {
  criterion = match.arg(criterion)
  plot = match.arg(plot)
  if(ncol(data) == 1) {
    if(plot == "marginals") plot = "density"
    return( mde(drop(data), bwf=bwf, tol=tol, criterion=criterion, 
                exhaustive=exhaustive, verbose=verbose, plot=plot, ...) )
  }
  bwf = unique(sort(bwf))
  k = length(bwf)
  ic = double(k)
  icmin = Inf
  names(icmin) = criterion
  n = nrow(data)                # sample size
  m = ncol(data)                # dimensionality
  if(missing(by)) by = which.min(colSums(apply(data, 2, duplicated)))
  info = matrix(nrow=k, ncol=3, dimnames=list(1:k,c("bwf", "ll", criterion)))
  for(i in 1:k) {
    r = cnmmb(data, bwf=bwf[k-i+1], by=by, tol=tol, ...)
    p = 2 * nrow(r$mix$mu) + m * (m+1) * 0.5 - 1
    ic[i] = -2 * r$ll +
      switch(criterion,
             AICc = {2 * p + 2 * p * (p + 1) / max(n - p - 1, 0)},
             AIC = {2 * p},
             BIC = {log(n) * p}
             )
    info[i,] = c(bwf[k-i+1], r$ll, ic[i])
    if(k == 1 || ic[i] < icmin ) {
      icmin[1] = ic[i]
      imin = i
      h = r$h
      mix = r$mix
      ll = r$ll
      maxgrad = r$max.gradient
    }
    if(plot != "null") plot(r$mix, data, type=plot)
    if(verbose >= 1)
      cat("Iter.#", i, ":\tbwf = ", sprintf("%.3f", bwf[k-i+1]), "\tll = ",
          r$ll, "\t", criterion, " = ", ic[i], "\n", sep="")
    if(verbose >= 2) print(r$mix)
    if(ic[i] == Inf || !exhaustive && ic[i] >= icmin + 10) break
  }
  fit = list(info=info[1:i,], maxgrad=maxgrad, ll=ll, criterion=icmin,
             mix=mix, h=h, bwf=bwf[k-imin+1])
  class(fit) = "mmde"
  fit
}

plot.mmde = function(x, data=NULL,
        type=c("marginals","density","components","gradient","pairs"),
        ...) {
  type = match.arg(type)
  plot(x$mix, data, type=type, ...)
  invisible(x)
}

predict.mmde = function(x, newdata, type=c("pdf","cdf"), lower.tail=TRUE,
                        log=FALSE) {
  type = match.arg(type)
  switch(type,
         pdf = dmvnmix(newdata, x$mix, log=log),
         cdf = pmvnmix(newdata, x$mix, lower.tail=lower.tail, log=log) )
}

