## =============================================== ##
## Functions for Multivariate Normal Distributions ##
##                                                 ##
## Author: Yong Wang (yongwang@auckland.ac.nz)     ##
##         Department of Statistics                ##
##         University of Auckland                  ##
##         New Zealand                             ##
## =============================================== ##

rmvn = function(n, mu=c(0,0), Sigma=diag(nrow=length(mu)), chol.Sigma=FALSE) {
  k = length(mu)
  if( n == 0 ) return( matrix(ncol=k, nrow=0) )
  z = rnorm(n*k)
  dim(z) = c(n,k)
  if(chol.Sigma) z %*% Sigma + rep(mu, rep(n, k))
  else z %*% chol(Sigma) + rep(mu, rep(n, k))
}

dmvn = function(x, mu=rep(0, ncol(x)), Sigma=diag(nrow=ncol(x)),
                chol.Sigma=FALSE, log=FALSE) {
  if(is.vector(x)) x = t(x)
  if(is.vector(mu)) {mu = rep(mu, rep(nrow(x), length(mu))); dim(mu) = dim(x)}
  if(nrow(x) == 1 && nrow(mu) > 1) x = x[rep(1,nrow(mu)),,drop=FALSE]
  if(ncol(x) != ncol(Sigma)) stop("x and Sigma non-conforming")
  if(nrow(Sigma) != ncol(Sigma)) stop("Sigma not a square matrix")
  if(ncol(mu) != nrow(Sigma)) stop("mu and Sigma non-conforming")
  if(!chol.Sigma) Sigma = chol(Sigma)
  dist2 = colSums(backsolve(Sigma, t(x - mu), transpose=TRUE)^2) 
  logd = -0.5 * (ncol(x)*1.8378770664093453 + 2*sum(log(diag(Sigma))) + dist2)
  if (log) logd
  else exp(logd)
}

pmvn = function(x, mu=rep(0, ncol(x)), Sigma=diag(nrow=ncol(x)),
                chol.Sigma=FALSE, lower.tail=TRUE, log=FALSE) {
  if(is.vector(x)) x = t(x)
  if(is.vector(mu)) {mu = rep(mu, rep(nrow(x), length(mu))); dim(mu) = dim(x)}
  if(nrow(x) == 1 && nrow(mu) > 1) x = x[rep(1,nrow(mu)),,drop=FALSE]
  if(ncol(x) != ncol(Sigma)) stop("x and Sigma non-conforming")
  if(nrow(Sigma) != ncol(Sigma)) stop("Sigma not a square matrix")
  if(ncol(mu) != nrow(Sigma)) stop("mu and Sigma non-conforming")
  if(!chol.Sigma) Sigma = chol(Sigma)
  z = backsolve(Sigma, t(x - mu), transpose=T)
  logp = colSums(pnorm(z, lower.tail=lower.tail, log.p=TRUE))
  if (log) logp
  else exp(logp)
}

# Outer operation between observations and means, with an idential
# covariance matrix for all means

outer.dmvn = function(x, mu, Sigma, chol.Sigma=FALSE, log=FALSE) {
  if(is.vector(x)) x = t(x)
  if(is.vector(mu)) mu = t(mu)
  n = nrow(x)
  m = nrow(mu)
  d = dmvn(x[rep(1:n, m),,drop=FALSE], mu[rep(1:m, rep(n,m)),,drop=FALSE],
           Sigma, chol.Sigma, log=log)
  dim(d) = c(n, m)
  d
}

